"use client";

import { motion } from "framer-motion";
import { MenuButtons } from "./MenuButtons";
import { ColumnButtons } from "./ColumnButtons";
import { useEffect, useRef, useState } from "react";
import {
  enumToTColumn,
  TCard,
  TCardProps,
  tcolumnToEnum,
} from "@/lib/types/kanban";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { TaskForm } from "@/features/task/components/task-form";
import { format, differenceInDays } from "date-fns";

export const Card = ({
  card,
  hanldeDragStart,
  updateCard,
  deleteCard,
}: TCardProps) => {
  const { id, title, column } = card;
  const [menuOpen, setMenuOpen] = useState(false);
  const [toMoveOpen, setToMoveOpen] = useState(false);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const cardRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (
        cardRef.current &&
        !cardRef.current?.contains(event?.target as Node)
      ) {
        setMenuOpen(false);
        setToMoveOpen(false);
      }
    };
    document.addEventListener("mousedown", handleClickOutside);
    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
  }, []);

  const handleDelete = () => {
    const confirm = window.confirm(`Are you sure you want delete this ?`);
    if (!confirm) return;
    deleteCard(card);
  };

  const handleMoveOpen = () => {
    setMenuOpen(false);
    setToMoveOpen(true);
  };

  const handleMove = (column: "pending" | "inprogress" | "completed") => {
    const newCard = { ...card, column };
    updateCard(newCard);
  };

  return (
    <>
      <motion.div
        layout
        layoutId={id}
        draggable
        ref={cardRef}
        onDragStart={(event) => hanldeDragStart(event, card)}
        className="cursor-grab rounded border border-neutral-700 bg-neutral-800 p-5 lg:p-3 my-2 active:cursor-grabbing relative"
        onClick={() => setIsDialogOpen(true)}
      >
        <div className="flex items-center gap-2 mb-2 flex-wrap">
          {card.priority && (
            <span
              className={`px-2 py-0.5 text-xs rounded ${
                {
                  high: "bg-red-600/20 text-red-400",
                  medium: "bg-orange-600/20 text-orange-400",
                  low: "bg-green-600/20 text-green-400",
                }[card.priority.toLowerCase()] ||
                "bg-neutral-600/20 text-neutral-400"
              }`}
            >
              {card.priority}
            </span>
          )}
          <div className="flex-grow"></div>
          {card.due_date &&
            (() => {
              const dueDate = new Date(card.due_date);
              const today = new Date();
              const daysRemaining = differenceInDays(dueDate, today);
              const dueDateStyles = {
                overdue: "bg-red-600/30 text-red-400",
                urgent: "bg-amber-600/30 text-amber-400",
                warning: "bg-orange-600/30 text-orange-400",
                normal: "bg-blue-600/30 text-blue-400",
              };
              const style =
                daysRemaining < 0
                  ? dueDateStyles.overdue
                  : daysRemaining === 0
                  ? dueDateStyles.urgent
                  : daysRemaining <= 3
                  ? dueDateStyles.warning
                  : dueDateStyles.normal;
              return (
                <span className={`px-2 py-0.5 text-xs rounded ${style}`}>
                  Due {format(dueDate, "MMM d")}{" "}
                  {daysRemaining >= 0 && `(${daysRemaining}d)`}
                </span>
              );
            })()}
        </div>
        <p className="text-sm text-neutral-100 break-words">{title}</p>
        <div>
          <span className={`px-2 py-0.5 text-xs rounded`}>
            {format(card.created_at, "MM dd, HH:mm")}
          </span>
        </div>
        {menuOpen && (
          <MenuButtons
            handleMoveOpen={handleMoveOpen}
            handleDelete={handleDelete}
          />
        )}
        {toMoveOpen && (
          <ColumnButtons column={column} handleMove={handleMove} />
        )}
      </motion.div>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Edit Task</DialogTitle>
          </DialogHeader>
          <TaskForm
            defaultValues={{
              ...card,
              status: tcolumnToEnum(column),
              user: card.user,
            }}
            onSubmit={(values) => {
              const tcard: TCard = {
                id: values.id,
                created_at: values.created_at,
                title: values.title,
                due_date: values.due_date,
                priority: values.priority,
                column: enumToTColumn(values.status),
                user: null,
              };
              updateCard(tcard);
              setIsDialogOpen(false);
            }}
            onDelete={handleDelete}
          />
        </DialogContent>
      </Dialog>
    </>
  );
};
