"use client";

import { getRoleName, UserRoleEnum } from "@/lib/enums/role";
import {
  MemberRoleInput,
  memberRoleSchema,
  OwnerRoleInput,
  ownerRoleSchema,
} from "@/lib/validations/role";
import { zodResolver } from "@hookform/resolvers/zod";
import { useRouter } from "next/navigation";
import { useState } from "react";
import { useForm } from "react-hook-form";
import {
  Tabs,
  TabsContent,
  TabsList,
  TabsTrigger,
} from "../../../components/ui/tabs";
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "../../../components/ui/card";
import { Label } from "../../../components/ui/label";
import { Input } from "../../../components/ui/input";
import { Button } from "../../../components/ui/button";
import { toast } from "sonner";
import { useUpdateUser } from "@/features/user/supabase/update-user";
import { useUser } from "@/lib/auth";
import { User } from "@/lib/types/user";

export function RoleForm() {
  const router = useRouter();
  const [isLoading, setIsLoading] = useState(false);
  const [selectedRole, setSelectedRole] = useState<UserRoleEnum>(
    UserRoleEnum.OWNER
  );

  const updateUser = useUpdateUser({
    mutationConfig: {
      onSuccess: () => {
        onUpdateUserSuccess();
      },
      onError: (error: Error) => {
        onUpdateUserError(error);
      },
    },
  });

  const user = useUser();

  const onUpdateUserSuccess = () => {
    toast.success("User updated successfully");
    if (selectedRole === "OWNER") {
      router.replace("/app/owner");
    } else if (selectedRole === "MEMBER") {
      router.replace("/app/member");
    } else {
      toast.error("Invalid role");
    }
    router.refresh();
  };

  const onUpdateUserError = (error: Error) => {
    toast.error(error.message);
  };

  const ownerForm = useForm<OwnerRoleInput>({
    resolver: zodResolver(ownerRoleSchema),
    defaultValues: {
      role: UserRoleEnum.OWNER,
      name: "",
    },
  });

  const memberForm = useForm<MemberRoleInput>({
    resolver: zodResolver(memberRoleSchema),
    defaultValues: {
      role: UserRoleEnum.MEMBER,
      name: "",
    },
  });

  async function onOwnerSubmit(data: OwnerRoleInput) {
    setIsLoading(true);
    const userId = user.data?.id;
    const email = user.data?.email;
    if (!userId && !email) {
      toast.error("Registered account is required");
      return;
    }
    const updateUserInput = {
      data: {
        id: userId!,
        email: email!,
        name: data.name,
        role: data.role,
        position: null,
      },
    };
    updateUser.mutate(updateUserInput);
    setIsLoading(false);
  }

  async function onMemberSubmit(data: MemberRoleInput) {
    setIsLoading(true);
    const userId = user.data?.id;
    const email = user.data?.email;
    if (!userId && !email) {
      toast.error("Registered account is required");
      return;
    }
    const updateUserInput = {
      data: {
        id: userId!,
        email: email!,
        name: data.name,
        role: data.role,
        position: null
      },
    };
    updateUser.mutate(updateUserInput);
    setIsLoading(false);
  }

  return (
    <div className="flex flex-col gap-4">
      <Tabs defaultValue={getRoleName(selectedRole)} className="w-full">
        <TabsList className="grid w-full grid-cols-2">
          <TabsTrigger value={getRoleName(UserRoleEnum.OWNER)}>
            Owner
          </TabsTrigger>
          <TabsTrigger value={getRoleName(UserRoleEnum.MEMBER)}>
            Member
          </TabsTrigger>
        </TabsList>
        <TabsContent value={getRoleName(UserRoleEnum.OWNER)}>
          <Card>
            <form onSubmit={ownerForm.handleSubmit(onOwnerSubmit)}>
              <CardHeader>
                <CardTitle>Owner</CardTitle>
                <CardDescription>
                  Finish your registration by adding your name.
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-2 gap-4 py-4">
                <div className="space-y-1">
                  <Label htmlFor="name">Name</Label>
                  <Input
                    id="name"
                    placeholder="Enter your name"
                    {...ownerForm.register("name")}
                  />
                </div>
              </CardContent>
              <CardFooter>
                <Button type="submit" disabled={isLoading}>
                  {isLoading ? "Saving..." : "Save"}
                </Button>
              </CardFooter>
            </form>
          </Card>
        </TabsContent>
        <TabsContent value={getRoleName(UserRoleEnum.MEMBER)}>
          <Card>
            <form onSubmit={memberForm.handleSubmit(onMemberSubmit)}>
              <CardHeader>
                <CardTitle>Member</CardTitle>
                <CardDescription>
                  Finish your registration by adding your name.
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-2 gap-4 py-4">
                <div className="space-y-1">
                  <Label htmlFor="name">Name</Label>
                  <Input
                    id="name"
                    type="text"
                    placeholder="Enter your name"
                    {...memberForm.register("name")}
                  />
                </div>
              </CardContent>
              <CardFooter>
                <Button type="submit" disabled={isLoading}>
                  {isLoading ? "Saving..." : "Save"}
                </Button>
              </CardFooter>
            </form>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
