"use client";

import { Board } from "./kanban";
import { useEffect, useState } from "react";
import { tasksToTCards, TCard, tcardToTask } from "@/lib/types/kanban";
import { useInsertTask } from "@/features/task/supabase/insert-task";
import { toast } from "sonner";
import { useUpdateTask } from "@/features/task/supabase/update-task";
import { useDeleteTask } from "@/features/task/supabase/delete-task";
import { TaskInput } from "@/lib/validations/task";
import { useGetUserTask } from "@/features/task/supabase/get-user-task";
import { useUser } from "@/lib/auth";

export const MemberBoard = () => {
  const user = useUser();
  const task = useGetUserTask();

  const insertTask = useInsertTask({
    mutationConfig: {
      onSuccess: handleInsertSuccess,
      onError: handleInsertError,
    },
  });
  const updateTask = useUpdateTask({
    mutationConfig: {
      onSuccess: handleUpdateSuccess,
      onError: handleUpdateError,
    },
  });
  const deleteTask = useDeleteTask({
    mutationConfig: {
      onSuccess: handleDeleteSuccess,
      onError: handleDeleteError,
    },
  });
  const [cards, setCards] = useState<TCard[]>([]);

  useEffect(() => {
    const tasks = task.data;
    if (tasks) {
      setCards(tasksToTCards(tasks));
    }
  }, [task.data]);

  function handleInsertSuccess() {
    toast.success("Task added successfully");
  }

  function handleInsertError(error: Error) {
    toast.error(error.message);
  }

  function handleUpdateSuccess() {
    toast.success("Task updated successfully");
  }

  function handleUpdateError(error: Error) {
    toast.error(error.message);
  }

  function handleDeleteSuccess() {
    toast.success("Task deleted successfully");
  }

  function handleDeleteError(error: Error) {
    toast.error(error.message);
  }

  function handleInsertTask(card: TCard) {
    const task = tcardToTask(card);
    const taskInput: TaskInput = {
      id: crypto.randomUUID(),
      title: task.title,
      status: task.status,
      priority: task.priority,
      due_date: task.due_date,
      created_at: new Date(),
      user: task.user ?? user.data!,
      column: card.column,
    };
    insertTask.mutate({ data: taskInput });
    setCards((prev) => [...prev, card]);
  }

  function handleUpdateTask(card: TCard) {
    const task = tcardToTask(card);
    const taskInput: TaskInput = {
      id: task.id,
      title: task.title,
      status: task.status,
      priority: task.priority,
      due_date: task.due_date,
      created_at: task.created_at,
      user: user.data!,
      column: card.column,
    };
    updateTask.mutate({ data: taskInput });
    setCards((prev) => prev?.map((c) => (c.id === card.id ? card : c)));
  }

  function handleDeleteTask(card: TCard) {
    deleteTask.mutate({ taskId: card.id });
    setCards((prev) => prev?.filter((c) => c.id !== card.id));
  }

  return (
    <Board
      cards={cards}
      insertCard={handleInsertTask}
      updateCard={handleUpdateTask}
      deleteCard={handleDeleteTask}
    />
  );
};
