import { QueryConfig } from "@/lib/react-query";
import { Task, TaskDto } from "@/lib/types/task";
import { User } from "@/lib/types/user";
import { createClient } from "@/utils/supabase/client";
import { supabaseConfig } from "@/utils/supabase/config";
import { queryOptions, useQuery } from "@tanstack/react-query";

export const getAllTask = async (): Promise<
  Map<string, { user: User; tasks: Task[] }>
> => {
  const supabase = createClient();
  const { data, error } = await supabase
    .from(supabaseConfig.column.task)
    .select(
      `
        id,
        created_at,
        title,
        status,
        due_date,
        priority,
        user ( id, name, email, role, position )
    `
    )
    .overrideTypes<Task[]>();
  if (error) {
    throw new Error(error.message);
  }

  // Sort tasks by status and priority
  const sortedData = data.sort((a, b) => {
    // First sort by status: In Progress -> Pending -> Completed
    const statusOrder = {
      "In Progress": 0,
      Pending: 1,
      Completed: 2,
    };
    const statusDiff = statusOrder[a.status] - statusOrder[b.status];
    if (statusDiff !== 0) return statusDiff;

    // Then sort by priority: High -> Medium -> Low -> None
    const priorityOrder = {
      High: 0,
      Medium: 1,
      Low: 2,
      None: 3,
    };
    return priorityOrder[a.priority] - priorityOrder[b.priority];
  });

  const taskMap = new Map<string, { user: User; tasks: Task[] }>();

  sortedData.forEach((task) => {
    const userId = task.user!.id;
    const existingEntry = taskMap.get(userId);

    if (existingEntry) {
      existingEntry.tasks.push(task);
    } else {
      taskMap.set(userId, {
        user: task.user!,
        tasks: [task],
      });
    }
  });

  const sortedTaskMap = new Map<string, { user: User; tasks: Task[] }>(
    Array.from(taskMap.entries()).sort(([, a], [, b]) => {
      // Handle cases where position might be undefined or null
      const posA = a.user.position ?? Number.MAX_SAFE_INTEGER; //  highest value if undefined.
      const posB = b.user.position ?? Number.MAX_SAFE_INTEGER;
      return posA - posB;
    })
  );

  return sortedTaskMap;
};

export const getAllTaskQueryOptions = () => {
  return queryOptions({
    queryKey: ["all-task"],
    queryFn: getAllTask,
  });
};

type UseGetAllTaskOptions = {
  queryConfig?: QueryConfig<typeof getAllTask>;
};

export const useGetAllTask = ({ queryConfig }: UseGetAllTaskOptions = {}) => {
  return useQuery({
    ...getAllTaskQueryOptions(),
    ...queryConfig,
  });
};
